/**********************************************************************
*	Bastion Products Copyright 1998								*
*	All rights reserved										*
*	This source is freely distributable in an unmodified state.		*
*	Project:			Blithe Static Library						*
*	Source:			Blithe.c									*
*	Last Modified:	5/20/98									*
*	Author:			Jennifer Weston							*
*	Description:		This project creates a static library which	*
*					provides weak linking to the Blithe shared	*
*					library and a C++ wrapper class. The Blithe.c	*
*					file provides the C weak linking.			*
***********************************************************************/
#include "BlitheDictionary.h"	// C++ header
#include "Blithe.h"			// C prototypes
#include <cstdio>
#include <cstdlib>
#include <PopUpMenu.h>
#include <Path.h>
#include <FindDirectory.h>
#include <Directory.h>
#include <MenuItem.h>
#include <Application.h>
#include <NodeInfo.h>
#include <cstring>
#include <Resources.h>
#include <Bitmap.h>

BResources *FindNextLocalDictionary(int32 inLangID,int32 i);
BResources *FindCommonDictionary(int32 inLangID);
BResources *FindDictionaryInPath(BPath thePath,int32 inLangID,int32 i);
int32 GetFirstDictionary(void);

/**********************************************************************
*	Method:		BlitheDictionary (constructor)					*
*	Arguments:	int32	dictionary							*
*							the language ID that is used as the	*
*							default language					*
*				bool		checkForAppDictionary					*
*							if true, checks if there is an		*
*							application dictionary for language	*
*							ID dictionary						*
*	Returns:		<none>										*
*	Description:	Initializes the C API, sets the default language,	*
*				and optionally checks to see if there is an		*
*				application dictionary for language ID dictionary	*
**********************************************************************/
BlitheDictionary::BlitheDictionary(int32 dictionary,bool checkForAppDictionary)
{
	status_t err = blithe_init();
	if (err != B_OK) return;
	appLanguage = (dictionary == BL_DEFAULT_LANGUAGE) ? get_default_language() : dictionary;
	temp = NULL;
	if (!checkForAppDictionary) return;

// Check for an app dictionary
	if (!has_application_dictionary(appLanguage))
//		be_app->PostMessage(BLITHE_NO_APP_DICTIONARY);
		throw BlitheException('NOAP');
}

/**********************************************************************
*	Method:		~BlitheDictionary (destructor)					*
*	Arguments:	<none>										*
*	Returns:		<none>										*
*	Description:	Deletes any allocated memory for the last entry	*
**********************************************************************/
BlitheDictionary::~BlitheDictionary(void)
{
	ClearTemp();
}

/**********************************************************************
*	Method:		AppLanguageID									*
*	Arguments:	<none>										*
*	Returns:		int32	The application language				*
*	Description:	Returns the application language					*
**********************************************************************/
inline int32 BlitheDictionary::AppLanguageID(void)
{
	return appLanguage;
}

/**********************************************************************
*	Method:		AppLanguageName								*
*	Arguments:	char*	inBuffer								*
*							Optional buffer for the name		*
*				int32	inBuffLength							*
*							Length of inBuffer in bytes			*
*	Returns:		char*	The application language name			*
*	Description:	Returns the name associated with the application	*
*				language ID. If inBuffer is not NULL, a maximum of	*
*				inBuffLength-1 characters is copied into inBuffer	*
*				and inBuffer is returned. Otherwise, memory for	*
*				the name is allocated. This memory should not be	*
*				deallocated by the user.						*
**********************************************************************/
inline char* BlitheDictionary::AppLanguageName(char *inBuffer,int32 inBuffLength)
{
	ClearTemp();
	if (inBuffer && inBuffLength) return language_name(appLanguage,inBuffer,inBuffLength);
	temp = language_name(appLanguage,NULL,0);
	return temp;
}

/**********************************************************************
*	Method:		Entry										*
*	Arguments:	int32	id									*
*							The requested entry ID				*
*				int32	dictionary							*
*							The requested language ID			*
*				char*	inBuffer								*
*							Optional buffer to store the string	*
*				int32	inBuffLength							*
*							Length of inBuffer in bytes			*
*	Returns:		char*	The string associated with entry			*
*						ID id and language ID dictionary			*
*	Description:	Finds the appropriate string for the entry ID and	*
*				and language ID. If inBuffer is not NULL, a maximum	*
*				of inBuffLength-1 characters is copied into		*
*				inBuffer and inBuffer is returned. Otherwise,		*
*				memory for the name is allocated. This memory		*
*				should not be deallocated by the user.			*
**********************************************************************/
char* BlitheDictionary::Entry(int32 id,int32 dictionary,char *inBuffer,int32 inBuffLength)
{
// Get the proper language ID
	switch (dictionary)
	{
	case BL_APP_LANGUAGE:
		dictionary = appLanguage;
		break;
	case BL_DEFAULT_LANGUAGE:
		dictionary = get_default_language();
		break;
	}

// Get the entry
	temp = (inBuffer != NULL && inBuffLength)	?	get_entry(id,dictionary,inBuffer,inBuffLength) : 
											get_entry(id,dictionary,NULL,0);
	char hexNum[32];
	sprintf(hexNum,"%lx",id);
//	if (strstr(temp,hexNum)) temp = static_get_entry(id,dictionary,inBuffer,inBuffLength);
	return temp;
}

/**********************************************************************
*	Method:		Blob											*
*	Arguments:	uint32	type									*
*							The requested resource type			*
*				int32	id									*
*							The requested resource ID			*
*				int32	dictionary							*
*							The requested language ID			*
*				size_t*	outBlobSize							*
*							size in bytes of the data contained	*
*							in the returned pointer				*
*	Returns:		void*	The data contained in the resource specified	*
*						by type, id and dictionary				*
*	Description:	Finds the appropriate resource for the type, id and	*
*				and language ID and returns a pointer to it.			*
*				This memory should not be deallocated by the user.	*
**********************************************************************/
void* BlitheDictionary::Blob(uint32 type,int32 id,size_t* outBlobSize,int32 dictionary, void* inBuffer, size_t inBuffLength)
{
// Get the proper language ID
	switch (dictionary)
	{
	case BL_APP_LANGUAGE:
		dictionary = appLanguage;
		break;
	case BL_DEFAULT_LANGUAGE:
		dictionary = get_default_language();
		break;
	}

// Get the blob
	void* t = get_blob(type,id,dictionary,inBuffer,inBuffLength,outBlobSize);

// If there was no buffer, this is our memory
	if (inBuffer == NULL || inBuffLength < 1)
	{
		ClearTemp();
		temp = (char*)t;
	}
	return t;
}

/**********************************************************************
*	Method:		LanguageList									*
*	Arguments:	char**	outList								*
*							Takes the address of a char* so it	*
*							can return an array which it		*
*							allocates							*
*	Returns:		int32	The number of names in the array			*
*	Description:	Calls through to get_language_list which gets the	*
*				name of each language and stores it.				*
*				in an array.									*
**********************************************************************/
inline int32 BlitheDictionary::LanguageList(char **outList)
{
	if (temp) free(temp);
	return get_language_list(outList);
}

/**********************************************************************
*	Method:		LanguageName									*
*	Arguments:	int32	id									*
*							The requested entry ID				*
*				char*	inBuffer								*
*							Optional buffer to store the name	*
*				int32 inBuffLength								*
*							Length of inBuffer in bytes			*
*	Returns:		int32	The string containing the name of		*
*						the application language				*
*	Description:	Gets the language name from	language_name.		*
**********************************************************************/
inline char* BlitheDictionary::LanguageName(int32 id,char *inBuffer,int32 inBuffLength)
{
	if (id == BL_DEFAULT_LANGUAGE) id = get_default_language();
	if (inBuffer && inBuffLength) return language_name(id,inBuffer,inBuffLength);
	temp = language_name(id,NULL,0);
	return temp;
}

/**********************************************************************
*	Method:		LanguageID									*
*	Arguments:	char*	name									*
*							The name of the language requested	*
*	Returns:		int32	The language ID						*
*	Description:	Gets the language ID from language_id.			*
**********************************************************************/
inline int32 BlitheDictionary::LanguageID(char *name)
{
	return language_id(name);
}

/**********************************************************************
*	Method:		ClearTemp										*
*	Arguments:	<none>										*
*	Returns:		<none>										*
*	Description:	Used internally to this class. It cleans up the	*
*				last memory allocation made by this object.		*
**********************************************************************/
void BlitheDictionary::ClearTemp(void)
{
	if (temp)
	{
		free(temp);
		temp = NULL;
	}
}

/**********************************************************************
*	Method:		LanguagePopupMenu								*
*	Arguments:	<none>										*
*	Returns:		BPopupMenu*	A popup menu of all available		*
*							languages							*
*	Description:	Creates a popup menu of all available languages.	*
**********************************************************************/
BPopUpMenu *BlitheDictionary::LanguagePopupMenu(void)
{
	BPopUpMenu *thePopup = new BPopUpMenu("");

// Get the common dictionary directory ($B_COMMON_ETC_DIRECTORY$/Blithe)
	status_t err;
	BPath thePath;
	err = find_directory(B_COMMON_ETC_DIRECTORY,&thePath);
	thePath.Append("Blithe");
	BDirectory theDirectory(thePath.Path());
	if (theDirectory.InitCheck() != B_OK)
	{
		BMenuItem *theItem;
		BMessage *theMessage = new BMessage(BLITHE_LANGUAGE_CHANGED);
		theMessage->AddInt32("langid",0);
		theItem = new BMenuItem("",theMessage);
		theItem->SetTarget(be_app);
		thePopup->AddItem(theItem);
		return thePopup;
	}
	theDirectory.Rewind();
	BEntry candidate;

// Get the current language id
	int32 currentID = get_default_language();

// For each dictionary found, add a menu item with the name of the language and
// a message returning the language id
	BMenuItem *theItem = NULL;
	bool found=false;
	while (theDirectory.GetNextEntry(&candidate,true) == B_OK)
	{
		BNode theNode(&candidate);
		char mimeType[1024];
		BNodeInfo(&theNode).GetType(mimeType);
		if (strncmp(mimeType,"application/x-vnd.Bastion-BlitheDict",36) == 0)
		{
			BFile theFile(&candidate,B_READ_ONLY);
			BResources theResource(&theFile);
			size_t len;
			char *name = NULL;
			int32 *id = NULL;
			name = (char *)(theResource.FindResource('CSTR',(int32)0,&len));
			if (!name) continue;
			id = (int32 *)(theResource.FindResource('LONG',1,&len));
			if (!id)
			{
				free(name);
				continue;
			}

		// Create the message
			BMessage *theMessage = new BMessage(BLITHE_LANGUAGE_CHANGED);
			theMessage->AddInt32("langid",*id);
			theItem = new BMenuItem(name,theMessage);
			theItem->SetTarget(be_app);
		// If the default language id equals the language id of the current file
			if (currentID == *id)
			{
				theItem->SetMarked(true);
				found = true;
			}
			free(name);
			free(id);
			thePopup->AddItem(theItem);
		}
	}
	if (!found) theItem->SetMarked(true);

	return thePopup;
}

/**********************************************************************
*	Method:		SetAppLanguage								*
*	Arguments:	int32	inLangID								*
*							The new application language ID		*
*	Returns:		<none>										*
*	Description:	Sets the application language ID to inLangID.		*
**********************************************************************/
void BlitheDictionary::SetAppLanguage(int32 inLangID)
{
	appLanguage = inLangID;
}

/**********************************************************************
*	Method:		BlitheBitmap									*
*	Arguments:	<none>										*
*	Returns:		BBitmap*	the Blithe bitmap						*
*	Description:	Gets the Blithe bitmap							*
**********************************************************************/
BBitmap* BlitheDictionary::BlitheBitmap(void)
{
	void* theBits;
	size_t size = get_raw_bits(&theBits);
	if (size == 0) return NULL;
	BBitmap *theBitmap = new BBitmap(BRect(0,0,63,63),B_COLOR_8_BIT,false,true);
	theBitmap->SetBits(theBits,size,0,B_COLOR_8_BIT);
	free(theBits);
	return theBitmap;
}

/**********************************************************************
*	Method:		FindNextLocalDictionary							*
*	Arguments:	int32	inLangID								*
*							the requested language ID			*
*				int32	i									*
*							index into the list of dictionaries	*
*	Returns:		BResources*	the BResource of the application		*
*				dictionary for language ID inLangID.				*
*	Description:	Looks for a file with the dictionary MIME type and	*
*				a language ID of inLangID. If found, returns the	*
*				BResource* for it. Otherwise returns NULL.		*
**********************************************************************/
BResources *FindNextLocalDictionary(int32 inLangID,int32 i)
{
	BPath thePath;
	app_info theAppInfo;
	status_t err = be_app->GetAppInfo(&theAppInfo);
	if (err != B_OK) return NULL;
	BEntry myEntry(&theAppInfo.ref,true);
	if (myEntry.InitCheck() != B_NO_ERROR) return NULL;
	BEntry parent;
	myEntry.GetParent(&parent);
	parent.GetPath(&thePath);
	thePath.Append("Blithe");
	return FindDictionaryInPath(thePath,inLangID,i);
}

/**********************************************************************
*	Method:		FindCommonDictionary							*
*	Arguments:	int32	inLangID								*
*							the requested language ID			*
*	Returns:		BResources*	the BResource of the common			*
*							dictionary for language ID inLangID.	*
*	Description:	Looks for a file with the dictionary MIME type and	*
*				a language ID of inLangID. If found, returns the	*
*				BResource* for it. Otherwise returns NULL.		*
**********************************************************************/
BResources *FindCommonDictionary(int32 inLangID)
{
	status_t err;
	BPath thePath;
	err = find_directory(B_COMMON_ETC_DIRECTORY,&thePath);
	if (err != B_NO_ERROR) return NULL;
	thePath.Append("Blithe");
	return FindDictionaryInPath(thePath,inLangID,0);
}

/**********************************************************************
*	Method:		FindDictionaryInPath							*
*	Arguments:	BPath	thePath								*
*							the path to search					*
*				int32	inLangID								*
*							the requested language ID			*
*	Returns:		BResources*	the BResource of the dictionary for	*
*							language ID inLangID.				*
*	Description:	Looks for a file with the dictionary MIME type and	*
*				a language ID of inLangID. If found, returns the	*
*				BResource* for it. Otherwise returns NULL.		*
**********************************************************************/
BResources *FindDictionaryInPath(BPath thePath,int32 inLangID,int32 i)
{
/* Take care of the default case */
	if (inLangID == BL_DEFAULT_LANGUAGE) inLangID = get_default_language();

// Get ready
	BDirectory theDirectory(thePath.Path());
	theDirectory.Rewind();
	BEntry candidate;
	int32 j=0;
	while (j<i && theDirectory.GetNextEntry(&candidate,true) == B_OK) ++j;

// Check each dictionary for the language id
	bool found=false;
	while (theDirectory.GetNextEntry(&candidate,true) == B_OK)
	{
		BNode theNode(&candidate);
		char mimeType[1024];
		BNodeInfo(&theNode).GetType(mimeType);
		if (strncmp(mimeType,"application/x-vnd.Bastion-BlitheDict",36) != 0) continue;

		BFile theFile(&candidate,B_READ_ONLY);
		if (theFile.InitCheck() != B_NO_ERROR) continue;
		BResources *theResource = new BResources(&theFile);
		if (!theResource) continue;
		size_t len;
		int32 *id;
		id = (int32 *)(theResource->FindResource('LONG',1,&len));
		if (!id) continue;
		found = (inLangID == *id);
		free(id);
		if (found) return theResource; 
		delete theResource;
	}
	return NULL;
}

/**********************************************************************
*	Method:		GetFirstDictionary								*
*	Arguments:	<none>										*
*	Returns:		int32	The first language ID it comes across in	*
*						the common dictionary directory.			*
*	Description:	Searches the common dictionaries for a language ID.	*
*				This is used in case we don't have a valid		*
*				language ID and we need one.					*
**********************************************************************/
int32 GetFirstDictionary(void)
{
// Get the path
	status_t err;
	BPath thePath;
	err = find_directory(B_COMMON_ETC_DIRECTORY,&thePath);
	if (err != B_NO_ERROR) return 0;
	thePath.Append("Blithe");
	BDirectory theDirectory(thePath.Path());
	theDirectory.Rewind();
	BEntry candidate;

// Get the first valid dictionary
	while (theDirectory.GetNextEntry(&candidate,true) == B_OK)
	{
		BNode theNode(&candidate);
		char mimeType[1024];
		BNodeInfo(&theNode).GetType(mimeType);
		if (strncmp(mimeType,"application/x-vnd.Bastion-BlitheDict",36) != 0) continue;

		BFile theFile(&candidate,B_READ_ONLY);
		if (theFile.InitCheck() != B_NO_ERROR) continue;
		BResources *theResource = new BResources(&theFile);
		if (!theResource) continue;
		size_t len;
		int32 *id;
		id = (int32 *)(theResource->FindResource('LONG',1,&len));
		if (!id) continue;

		int32 val = *id;
		free(id);
		return val;
	}
	return BL_ENGLISH_LANGUAGE;
}